/*
 * Name:	James Sheets
 * Program:	#2
 * Due Date:	6/6/06
 * Description:	Create a game using MinMax with alpha-beta pruning
 *		where the player and computer take turns picking numbers from a
 *		given range (0 to HIGHNUMBER) to get exactly X numbers
 *		(MATCHES) to sum to exactly Y (MATCHNUMBER).  MAXDEPTH is the
 *		number of minmax operations to perform.
 * Compiling:	Tested to work with Java 5.0 and Java 6.0
 *		On windows, if the /bin directory for your JDK is in your
 *		PATH directory, you can simply use the build.bat file to build.
 *		Otherwise, you will need to move to the directory program.java 
 *		exists in, and from a command prompt enter:
 *		PATH_TO_JDK_BIN\javac program.java
 * Run:		On windows, you can simply use the run.bat file to execute.
 *		Otherwise, you will need to move to the programs directory, and
 *		from a command prompt enter:
 *		java MinMax
 */
import java.io.*;






/*
 * This is our main class, where most of our work is done
 */
class MinMax
{
   
   /*
    * Global variables
    */
   public int MAXDEPTH;
   public int HIGHNUMBER;
   public int MATCHNUMBER;
   public int MATCHES;
   public String computerName;
   public String playerName;
   public Game game;
   
   
   /*
    * Program entry-point
    */
   public static void main(String[] args)
   {
      // Create an instance of our class
      MinMax mm = new MinMax();
      // Exit program
      System.exit(1);
   }
   
   
   
   /*
    * Default Constructor
    */
   public MinMax()
   {
      MAXDEPTH = 2;
      HIGHNUMBER = 9;
      MATCHNUMBER = 15;
      MATCHES = 3;
      
      computerName = "Computer";
      playerName = "Player";
      
      game = new Game(HIGHNUMBER);
      
      // Print intro
      System.out.println("Hello.  Welcome to the game.");
      System.out.println("Rules:  You want to sum exactly " + MATCHES + " numbers together that total to exactly " + MATCHNUMBER + ".");
      System.out.println("        You may pick from any of the unused numbers displayed below.");
      System.out.println("        The game ends when someone wins, or when there are no more moves to make");
      System.out.println();
      game.print();
      System.out.println();
      
      
      String userInput;
      BufferedReader br;
      // Loop until the game is done
      while (!isGameOver(game))
      {
	 // Ask the user to input a number
	 System.out.print("Enter a number: ");
	 br = new BufferedReader(new InputStreamReader(System.in));
	 try 
	 {
	    // Convert the input into a number
	    userInput = br.readLine();
	    int input = new Integer(userInput).intValue();
	    
	    // Make sure the number is within our range of valid numbers
	    if ((input < 1) || (input > HIGHNUMBER))
	    {
	       System.out.println("That number is out of range [0 - " + HIGHNUMBER + "]");
	       continue;
	    }
	    
	    // Make sure the number isn't taken
	    if (game.getOwner(input-1) != null)
	    {
	       System.out.println("That number is already taken.");
	       continue;
	    }
	    
	    // If all is good, give the player the number
	    game.setOwner(input-1,playerName);
	    // Check if player won
	    if (isGameOver(game))
	       break;
	    
	    // Generate our move
	    Move move = maxMove(game,0,new Move(-999), new Move(999));
	    game.setOwner(move.position,computerName);
	    game.print();
	    System.out.println();
	 } 
	 // We have to catch any IO exceptions
	 catch (IOException ioe) 
	 {
	    System.out.println("IO error occured.  Game exiting");
	    System.exit(1);
	 }
	 // Catch any attempts to convert a non-number
	 catch (NumberFormatException nfe)
	 {
	    System.out.println("That's not a valid number");
	    continue;
	 }
      }
      
      
      // Game's over, see what the outcome is
      game.print();
      System.out.println();
      if (weHaveWinner(playerName,game,0,0,0))
	 System.out.println("You win!");
      else if (weHaveWinner(computerName,game,0,0,0))
	 System.out.println("Sorry, you lose!");
      else
	 System.out.println("Stalemate");
   }
   
   
   
   /*
    * Performs our min calculation, with pruning
    */
   public Move minMove(Game gameInstance, int depth, Move alpha, Move beta)
   {
      if (isGameOver(gameInstance) || depth > MAXDEPTH)
      {
	 // Return the value of this game
	 return evaluateGame(gameInstance);
      }
      
      
      // We'll create a divergent possibily for every possible move left for player
      int lastMove = 0;
      for (int i=0; i<movesFor(null,gameInstance); i++)
      {
	 // Create a copy of the current game
	 Game divergentGame = gameInstance.copy();
	 // Create a new divergent game, with a new move for this player
	 lastMove = applyMove(playerName,divergentGame,lastMove);
	 // Find the max move
	 Move move = maxMove(divergentGame, depth+1, alpha, beta);
	 
	 // If the player's found a better move, store it
	 if (move.value < beta.value)
	 {
	    
	    beta = move;
	    
	    // Prune
	    if (alpha.value >= beta.value)
	       return beta;
	 }
      }
      
      return beta;
   }
   
   
   
   /*
    * Performs our max calculation, with pruning
    */
   public Move maxMove(Game gameInstance, int depth, Move alpha, Move beta)
   {
      if (isGameOver(gameInstance) || depth > MAXDEPTH)
      {
	 // Return the value of this game
	 return evaluateGame(gameInstance);
      }
      
      
      // We'll create a divergent possibily for every possible move left for us
      int lastMove = 0;
      for (int i=0; i<movesFor(null,gameInstance); i++)
      {
	 // Create a copy of the current game
	 Game divergentGame = gameInstance.copy();
	 // Create a new divergent game, with a new move for this player
	 lastMove = applyMove(computerName,divergentGame,lastMove);
	 // Find the min move
	 Move move = maxMove(divergentGame, depth+1, alpha, beta);
	 
	 // If the computer has a better move, store it
	 if (move.value > alpha.value)
	 {
	    
	    alpha = move;
	    
	    // Prune
	    if (alpha.value >= beta.value)
	       return alpha;
	 }
      }
      
      return alpha;
   }
   
   
   
   /*
    * This simply returns the last move made by the player for a game
    */
   public Move evaluateGame(Game gameInstance)
   {
      // Create an instance of our Move class.  We will use this to associate a
      // value with the best first move we find
      Move move = new Move();
      
      // If the computer wins, give a high value
      if (weHaveWinner(computerName,gameInstance,0,0,0))
	 move.value = 1;
      // If the player wins, give a low value
      else if (weHaveWinner(playerName,gameInstance,0,0,0))
	 move.value = -1;
      // If it's [still] a tie, give a neutral value
      else
	 move.value = 0;
      
      // This will return the first move we made in this divergent game
      int lastIndex = 0;
      for (int i=0; i < HIGHNUMBER; i++)
      {
	 // the actual moves don't have a minMaxInsertOrder, so if we find 
	 // anything with a value other than 0, it was from a simulation
	 if (lastIndex == 0)
	 {
	    if (gameInstance.minMaxInsertOrder[i] > 0)
	       lastIndex = i;
	 }
	 // If we find any minMaxInsertOrder number less than our current one, 
	 // it occured before the current step.  Do this till we get the first
	 // move we made in this divergent game
	 else
	 {
	    if (gameInstance.minMaxInsertOrder[i] > gameInstance.minMaxInsertOrder[lastIndex] &&
	       gameInstance.minMaxInsertOrder[i] != 0)
	    	lastIndex = i;
	 }
      }
      
      move.position = lastIndex;
      
      // Return the move
      return move;
   }
   
   
   /*
    * See how many possible moves are left
    */
   public int movesFor(String who, Game gameInstance)
   {
      int movesLeft = 0;
      for (int i=0; i < gameInstance.length; i++)
	 if (gameInstance.getOwner(i) == who)
	    ++movesLeft;
      
      return movesLeft;
   }
   
   
   /*
    * Apply the first possible move (after lastMove) to the game for the
    * specified player to the game instance  We use this function to create 
    * game simulations by finding possible moves
    */
   public int applyMove(String who, Game gameInstance, int lastMove)
   {
      for (int i=lastMove; i < gameInstance.length; i++)
      {
	 if (gameInstance.getOwner(i) == null)
	 {
	    gameInstance.setOwner(i,who,i+1);
	    
	    // Incremeant the return value to the next [possible] open move
	    return i+1;
	 }
      }
      
      return lastMove;
   }
   
   
   
   /*
    * Checks if the game should end because someone's won, or because
    * we've run out of numbers
    */
   public boolean isGameOver(Game gameInstance)
   {
      // First check if the player has won
      if (weHaveWinner(playerName,gameInstance,0,0,0))
	 return true;
      
      // Next check if the computer has won
      if (weHaveWinner(computerName,gameInstance,0,0,0))
	 return true;
      
      // Next check if we're out of numbers
      for (int i=0; i < gameInstance.length; i++)
	 if (gameInstance.getOwner(i) == null)
	    return false;
      
      // If noone's won, and we didn't find an unused number, game is over
      return true;
   }
   
   
   
   /*
    * Check to see if the player or computer has won the game
    */
   public boolean weHaveWinner(String who, Game gameInstance, int offset, int iter, int total)
   {
      // Don't allow more than X numbers to be added together
      if (iter > MATCHES)
	 return false;
      
      // Loop through every possible number
      for (int i = offset; i < HIGHNUMBER; i++)
      {
	 // See if the owner is the player we're checking against
	 if (gameInstance.getOwner(i) == who)
	 {
	    // If we've summed exactly X numbers together, we can check to see
	    // if we have a match
	    if ( iter+1 == MATCHES && total+i+1 == MATCHNUMBER )
	       return true;
	    
	    // Otherwise, perform a recursive search to see if this number can
	    // be added with any others to create a match
	    if ( weHaveWinner(who, gameInstance, i+1, iter+1, total+i+1) )
	       return true;
	 }
      }
      
      // If all else fails, return false
      return false;
   }
   
   
   /*
    * This will get the lowest value number combination available for a player
    */
   /*public int lowestPossibleTotal(String who, Game gameInstance, int offset, int iter, int total)
   {
      // We've reached our limits for this combination, return the total
      if (iter == MATCHES || iter > movesFor(who,gameInstance))
	 return total;
      
      // Assign a high value that we shouldn't hit
      int testTotal = 999;
      // Loop through every value
      for (int i=offset; i < HIGHNUMBER; i++)
      {
	 // If this is a value owned by the player
	 if (gameInstance.getOwner(i) == who)
	 {
	    // Use recursion to get all possible sum combinations using this number
	    int tempTotal = lowestPossibleTotal(who,gameInstance,i+1,iter+1,total+i+1);
	    // If the sum is less than the current total, we'll use it (for now)
	    if (tempTotal < testTotal)
	       testTotal = tempTotal;
	 }
      }
      
      return testTotal;
   }*/
   
}



/*
 * Simple class which will allow us to assign values for moves.  This way, we
 * when we evalute a simulated game, we can just pass back one variable without
 * re-evaluating any game simulation
 */
class Move
{
   public int value;
   public int position;
   
   /*
    * Default Constructor
    */
   public Move()
   {
      value = 0;
      position = 0;
   }
   
   /*
    * Overloaded Constructor (used when making an alpha or beta)
    */
   public Move(int i)
   {
      value = i;
      position = 0;
   }
}



/*
 * This class is used to hold the state of our game, or for possible divergent
 * games.  Each possible number will be equal to the index -1 of our values
 * array, which holds the name of the player who holds that number
 */
class Game
{
   public String[] values;
   public int length;
   public int[] minMaxInsertOrder; // We can use this to keep track of numbers we insert during minmax scanning
   
   
   /*
    * Overloaded constructor to initialize our arrays
    */
   public Game(int size)
   {
      length = size;
      values = new String[size];
      minMaxInsertOrder = new int[size];
   }
   
   /*
    * Returns the name of the person who's assigned to a number
    */
   public String getOwner(int index)
   {
      return values[index];
   }
   
   /*
    * Sets the owner of a number
    */
   public void setOwner(int index, String owner)
   {
      if (index >=0 && index < length)
	 values[index] = owner;
   }
   
   /*
    * Sets the owner of a number, as well as the order the number was picked
    * We use this for divergent simulations, so once we have found the optimal
    * simulation, we can pick the last move from it that we should play
    */
   public void setOwner(int index, String owner, int insertOrder)
   {
      values[index] = owner;
      minMaxInsertOrder[index] = insertOrder;
   }
   
   /*
    * Returns a new copy of our game.  We use this when we want to create a 
    * divergent game for simulation
    */
   public Game copy()
   {
      Game g = new Game(values.length);
      for (int i=0; i < values.length; i++)
	 g.setOwner(i, values[i], minMaxInsertOrder[i]);
      
      return g;
   }
   
   /*
    * Prints out the current status of our game
    */
   public void print()
   {
      for (int i=1; i <= values.length; i++)
      {
	 System.out.println("[" + i + "] " + values[i-1]);
      }
   }
}